/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Tacho Resource Framework API Header File
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Dependants  :   MPC5xx OS sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_Tacho.h
    \brief A tachometer output is described by the BEHAVIOUR_TACHO behaviour. See \ref tachodefn. */

#ifndef __RESOURCE_TACHO_H
#define __RESOURCE_TACHO_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
/*! \brief The value of tachometer dutycycle that equates to 100%.

    Use this value to scale the application's scaling of percentage to get the units required by the framework
    \code
    uDutyCycle = (uAppPcent * FULL_SCALE_TACHO_DUTYCYCLE) / uAppPcentValueFor100; \endcode */
#define FULL_SCALE_TACHO_DUTYCYCLE 4096L


/*! \brief The resolution of tachometer crank angle attributes. 1 degree crank angle equates to this value.

    Use this value to scale the application's scaling of crank engle to get the units required by the framework
    \code
    sStartAngle = (sStartAngle * TACHO_ANGLE_RESOLUTION) / 1; \endcode */
#define TACHO_ANGLE_RESOLUTION     16L

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the allowed pulses per revolution supported by this behaviour.

Note that it is pulse per engine revolution and not pulses per engine cycle. */
/* Name: E_TachoPulsePerRev ClassID:ENUMDEC EnumDflt:"Illegal" */
typedef enum
{
/*!< 1 Pulse per rev */     TAC_1_PULSE = 1,  /* EnumTxt:"1 Pulse per rev" */  
/*!< 2 Pulses per rev */    TAC_2_PULSE = 2,  /* EnumTxt:"2 Pulses per rev" */  
/*!< 3 Pulses per rev */    TAC_3_PULSE = 3,  /* EnumTxt:"3 Pulses per rev" */  
/*!< 4 Pulses per rev */    TAC_4_PULSE = 4,  /* EnumTxt:"4 Pulses per rev" */  
/*!< 5 Pulses per rev */    TAC_5_PULSE = 5,  /* EnumTxt:"5 Pulses per rev" */  
/*!< 6 Pulses per rev */    TAC_6_PULSE = 6,  /* EnumTxt:"6 Pulses per rev" */  
/*!< 7 Pulses per rev */    TAC_7_PULSE = 7,  /* EnumTxt:"7 Pulses per rev" */  
/*!< 8 Pulses per rev */    TAC_8_PULSE = 8,  /* EnumTxt:"8 Pulses per rev" */  
} E_TachoPulsePerRev;

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid Tachometer
           output attributes in the S_TachoResourceAttributes and S_TachoCreateResourceAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_TachoResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and report condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_TACHO_DUTYCYCLE;
    AttribsObj.u2DutyCycle = (50 * FULL_SCALE_TACHO_DUTYCYCLE) / 100L;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_TAC_CONDITION = 0x01,           /*!< Selects S_TachoResourceAttributes::eResourceCondition */
    USE_TAC_START_ANGLE = 0x02,         /*!< Selects S_TachoResourceAttributes::s2StartAngle */
    USE_TAC_DUTYCYCLE = 0x04,           /*!< Selects S_TachoResourceAttributes::u2DutyCycle */
    USE_TAC_NUM_PULSES = 0x08,          /*!< Selects S_TachoCreateResourceAttributes::ePulsesPerRev */
    USE_TAC_DYNAMIC_ON_CREATE = 0x10,   /*!< Selects S_TachoCreateResourceAttributes::DynamicObj */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_TachoAttributesMask;

/*! \brief This data structure describes all of the tachometer resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_TACHO(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_TachoResourceAttributes TachoObj;

    // Only going to alter the duty cycle member
    TachoObj.uValidAttributesMask = USE_TAC_DUTYCYCLE;  \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_TAC_CONDITION, \ref USE_TAC_START_ANGLE and \ref USE_TAC_DUTYCYCLE] that
    are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;
/*! \brief The start angle for the first pulse in the tachometer pulse stream in x16 degrees before firing TDC#1.

    Select this attribute with the \ref USE_TAC_START_ANGLE bit mask. The default value of this attribute upon resource
    creation is 0 degBfTDC#1. See \ref tachousageexample and \ref tachostartup. */
    sint2 s2StartAngle;
/*! \brief The percentage that the driver should be ON for in any pulse period.

    Use the \ref FULL_SCALE_TACHO_DUTYCYCLE to scale the duty cycle.

    Note that only values between 10% and 90% are acceptable. Supplying a value outside this range will be clipped by the framework.

    Select this attribute with the \ref USE_TAC_DUTYCYCLE bit mask. This attribute defaults to 50% upon resource creation.

    Also see \ref tachousageexample. */
    uint2 u2DutyCycle;
/*! \brief The condition of the resource. A resource with a \ref RES_DISABLED condition will not be active, even if there is engine
           speed present. Pulses will only occur when the condition is set to \ref RES_ENABLED.

    Select this attribute with the \ref USE_TAC_CONDITION bit mask. The default value of this attribute upon resource
    creation is \ref RES_DISABLED. Also see \ref tachousageexample for further usage information. */
    E_ResourceCond eResourceCondition;
} S_TachoResourceAttributes;

typedef S_TachoResourceAttributes const* S_TachoResourceAttributesPtr;

/*! \brief This data structure describes the creation attributes for Tachometer output resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_TACHO() function and \ref USE_TAC_DYNAMIC_ON_CREATE is included in the
    S_TachoCreateResourceAttributes::uValidAttributesMask member.
    
    \code
    ...
    S_TachoCreateResourceAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_TAC_DYNAMIC_ON_CREATE;
    // Only going to set the select the duty cycle member in the dynamic object
    CreateObj.DynamicObj.uValidAttributesMask = USE_TAC_DUTYCYCLE;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_TAC_NUM_PULSES and \ref USE_TAC_DYNAMIC_ON_CREATE] that
    are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;
/*! \brief The number of pulses per engine \b revolution that are to be generated by this resource.

    The pulses per engine revolution attribute defaults to \ref TAC_1_PULSE when this attribute is not selected via the
    \ref USE_TAC_NUM_PULSES bit mask. The number of pulses that can be generated is limited to the encoder and
    will be clipped to this limit. */
    E_TachoPulsePerRev ePulsesPerRev;

/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_TACHO(). Select with the \ref USE_TAC_DYNAMIC_ON_CREATE bit mask */
    S_TachoResourceAttributes DynamicObj;
} S_TachoCreateResourceAttributes;

typedef S_TachoCreateResourceAttributes const* S_TachoCreateResourceAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
/* Optimised calls that can be applied to certain attribute sets                                           */

/* DO NOT use these functions directly. They are implicitly called through the use of the CreateResource() */
/*   and SetResourceAttributes() macro functions                                                           */
NativeError_S CreateResourceBEHAVIOUR_TACHO(E_ModuleResource, S_TachoCreateResourceAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_TACHO(E_ModuleResource);
NativeError_S SetResourceAttributesBEHAVIOUR_TACHO(E_ModuleResource, S_TachoResourceAttributes const*);

#endif /* __RESOURCE_TACHO_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
